%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(100,50,0.5,1.5)
function SEMDOT2D(nelx,nely,vol,rmin)
%% INITIALIZATION
vx=repmat(vol,nely,nelx); vxPhys=vx;
Emin=0.001; maxloop=1000; nG=10; ngrid=nG-1; tolx=0.001;
E0=1; nu=0.3;
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% INITIALIZE HEAVISIDE SMOOTH FUNCTION
beta=0.5; ER=0.5;
%% PREPARE FINITE ELEMENT ANALYSIS
A11 = [12  3 -6 -3;  3 12  3  0; -6  3 12 -3; -3  0 -3 12];
A12 = [-6 -3  0  3; -3 -6 -3 -6;  0 -3 -6  3;  3 -6  3 -6];
B11 = [-4  3 -2  9;  3 -4 -9  4; -2 -9 -4 -3;  9  4 -3 -4];
B12 = [ 2 -3  4 -9; -3  2  9 -2;  4  9  2  3; -9 -2  3  2];
KE=1/(1-nu^2)/24*([A11 A12;A12' A11]+nu*[B11 B12;B12' B11]); 
nodenrs=reshape(1:(1+nelx)*(1+nely),1+nely,1+nelx);
edofVec=reshape(2*nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat=repmat(edofVec,1,8)+repmat([0 1 2*nely+[2 3 0 1] -2 -1],nelx*nely,1);
iK=reshape(kron(edofMat,ones(8,1))',64*nelx*nely,1);
jK=reshape(kron(edofMat,ones(1,8))',64*nelx*nely,1);
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely);
[fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% DEFINE LOADS AND SUPPORTS (CANTILEVER BEAM)
F=sparse(2*(nely+1)*(nelx+1)-nely,1,-1,2*(nely+1)*(nelx+1),1);
U=zeros(2*(nely+1)*(nelx+1),1);
fixeddofs=(1:2*(nely+1)); alldofs=[1:2*(nely+1)*(nelx+1)];
freedofs=setdiff(alldofs,fixeddofs); 
change=1;
loop=0;
%% START ITERATION
while (change > tolx || tol>0.001) && loop < maxloop
    loop=loop+1;
    %% FE-ANALYSIS
    sK=reshape(KE(:)*(vxPhys(:)'*E0+(1-vxPhys(:))'*(Emin*E0)),64*nelx*nely,1);
    K=sparse(iK,jK,sK); K=(K+K')/2; 
    U(freedofs)=K(freedofs,freedofs)\F(freedofs); 
    %% OBJECTIVE FUNCTION AND SENSITIVITY ANALYSIS
    ce=reshape(sum((U(edofMat)*KE).*U(edofMat),2),nely,nelx);
    c=sum(sum((vxPhys.*E0+(1-vxPhys).*(Emin*E0)).*ce));
    dc=-((1-vxPhys)*Emin+vxPhys).*E0.*ce; 
    dv=ones(nely,nelx);
    %% FILTERING/MODIFICATION OF SENSITIVITIES
    dc(:)=H*(dc(:)./Hs);
    dv(:)=H*(dv(:)./Hs);
    %% OPTIMALITY CRITERIA UPDATE OF DESIGN VARIABLES
    ll1 = 0; ll2 = 1e9; move = 0.2;
    while (ll2-ll1)/(ll1+ll2) > 1e-3
        lmid = 0.5*(ll2+ll1);
        vxnew = max(0,max(vx-move,min(1,min(vx+move,vx.*sqrt(-dc./dv/lmid)))));
        vxPhys(:)=(H*vxnew(:))./Hs;
        if sum(vxPhys(:)) > vol*nelx*nely, ll1 = lmid; else ll2 = lmid; end
    end
    xn=reshape((Hn*vxPhys(:)./Hns),nely+1,nelx+1);
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH FUNCTION
    xg=interp2(nodex,nodey,xn,fnx,fny,'linear'); 
    l1=0; l2=1;
    while (l2-l1)>1e-5
        ls=(l1+l2)/2;
        xgnew=max(0.001,(tanh(beta*ls)+tanh(beta*(xg-ls)))/(tanh(beta*ls)+tanh(beta*(1-ls))));
        if sum(sum(xgnew))/((ngrid*nelx+1)*(ngrid*nely+1))-sum(vxPhys(:))/...
                (nelx*nely)>0, l1=ls; else, l2=ls; end
    end
    %% ASSEMBLE GRID POINTS TO ELEMENTS
    vxPhys(:)=0;
    Terr=0; Tm=[];
    for i=1:nelx
        for j=1:nely
            e=(i-1)*nely+j;
            for i1=ngrid*(i-1)+1:ngrid*i+1
                for j1 = ngrid*(j-1)+1:ngrid*j+1
                    Tm = [Tm;xgnew(j1,i1)];
                    vxPhys(e)=vxPhys(e)+xgnew(j1,i1);
                end
            end
            if min(Tm)>0.001 && max(Tm)<1, Terr=Terr+1; end
            Tm=[];
        end
    end
    vxPhys=vxPhys/(ngrid+1)^2; 
    %% CHECK CONVERGENCE
    change=sum(abs(vxnew(:)-vx(:)))/(vol*nely*nelx);
    tol=Terr/(nely*nelx);
    vx=vxnew; 
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%8.4f Vol.:%4.3f ch.:%4.5f Topo.:%7.5f Beta.:%g.\n',...
        loop,c,mean(vxPhys(:)),change,tol,beta);
    contourf(fnx, flipud(fny), xg-ls, [0 0]);
    axis equal tight off; drawnow
    %% UPDATE HEAVISIDE SMOOTH FUNCTION
    beta=beta+ER;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% The basic structure of the code is based on the 88-line code from:      %
% "Efficient topology optimization in MATLAB using 88 lines of code",     %
%  E Andreassen, A Clausen, M Schevenels, BS Lazarov and O Sigmund,       %
%  SMO 2010, doi:10.1007/s00158-010-0594-7                                %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%








